"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createStrictAPI = void 0;
const error_1 = require("../utils/error");
const xcss_prop_1 = require("../xcss-prop");
/**
 * ## Create Strict API
 *
 * Returns a strict subset of Compiled APIs augmented by a type definition.
 * This API does not change Compileds build time behavior — merely augmenting
 * the returned API types which enforce:
 *
 * - all APIs use object types
 * - property values declared in the type definition must be used (else fallback to defaults)
 * - a strict subset of pseudo states/selectors
 * - unknown properties to be a type violation
 *
 * To set up:
 *
 * 1. Declare the API in a module (either local or in a package), optionally declaring accepted media queries.
 *
 * @example
 * ```tsx
 * // ./foo.ts
 * interface Schema {
 *   color: 'var(--ds-text)',
 *   '&:hover': { color: 'var(--ds-text-hover)' }
 * }
 *
 * const { css, cssMap, XCSSProp, cx } = createStrictAPI<Schema, { media: '(min-width: 30rem)' }>();
 *
 * export { css, cssMap, XCSSProp, cx };
 * ```
 *
 * 2. Configure Compiled to pick up this module:
 *
 * @example
 * ```diff
 * // .compiledcssrc
 * {
 * +  "importSources": ["./foo.ts"]
 * }
 * ```
 *
 * 3. Use the module in your application code:
 *
 * @example
 * ```tsx
 * import { css } from './foo';
 *
 * const styles = css({ color: 'var(--ds-text)' });
 *
 * <div css={styles} />
 * ```
 */
function createStrictAPI() {
    return {
        css() {
            throw (0, error_1.createStrictSetupError)();
        },
        cssMap() {
            throw (0, error_1.createStrictSetupError)();
        },
        cx: xcss_prop_1.cx,
        XCSSProp() {
            throw (0, error_1.createStrictSetupError)();
        },
    };
}
exports.createStrictAPI = createStrictAPI;
//# sourceMappingURL=index.js.map