/**
 * Creates a new function that will cache the result of it's first call.
 *
 * ```ts
 * function sayHello(name: string): string {
 *   return `Hello ${name}`;
 * }
 * const cached = once(sayHello);
 *
 * cached('Alex'); // returns "Hello Alex"
 * cached('Sam'); // returns "Hello Alex" (underlying `sayHello` function not called)
 * ```
 *
 * **Notes**
 *
 * - If the `onced` function throws, then the return value of the function is not cached
 * - Respects call site context (`this`) when executing the onced function
 */
export default function once(fn) {
  let cache = null;
  return function result(...args) {
    if (!cache) {
      cache = {
        value: fn.call(this, ...args)
      };
    }

    /**
     * Intentionally not adding `.clear()` function property
     *
     * - We currently have no need for it
     * - We don't want to add a `.clear()` to functions that should never be called twice,
     *   for example: `cleanup` functions.
     * - We can add a `onceWithCleanup` variant as a separate export from ds-lib if the
     *   need arises 🧘
     */

    return cache.value;
  };
}