import _defineProperty from "@babel/runtime/helpers/defineProperty";
import { _getFullUserHash, _getStorageKey, DataAdapterCachePrefix, DataAdapterCore, StableID } from '@statsig/js-client';

/**
 * Data adapter which only uses bootstrap data and will never fetch from network or cache.
 * We do this because we control the fetching of bootstrap data from FFS in Client.ts whereas the
 * default data adapter fetches from Statsig servers.
 */
export class NoFetchDataAdapter extends DataAdapterCore {
  constructor() {
    super('NoFetchDataAdapter', 'nofetch');
    _defineProperty(this, "bootstrapResult", null);
  }

  /**
   * Make sure to call this **before** calling `initializeAsync` or `updateUserAsync` but
   * after the Statsig client has been created!
   */
  setBootstrapData(data) {
    this.bootstrapResult = data ? {
      source: 'Bootstrap',
      data: JSON.stringify(data),
      receivedAt: Date.now(),
      stableID: StableID.get(this._getSdkKey()),
      fullUserHash: null
    } : null;
  }
  async prefetchData(_user, _options) {}
  async getDataAsync(_current, user, _options) {
    return this.bootstrapResult && {
      ...this.bootstrapResult,
      fullUserHash: _getFullUserHash(user)
    };
  }
  getDataSync(user) {
    return this.bootstrapResult && {
      ...this.bootstrapResult,
      fullUserHash: _getFullUserHash(user)
    };
  }
  async _fetchFromNetwork(_current, _user, _options) {
    return null;
  }
  _getCacheKey(user) {
    // Same logic as default data adapter
    // https://github.com/statsig-io/js-client-monorepo/blob/main/packages/js-client/src/StatsigEvaluationsDataAdapter.ts
    const key = _getStorageKey(this._getSdkKey(), user);
    return `${DataAdapterCachePrefix}.${this._cacheSuffix}.${key}`;
  }
  _isCachedResultValidFor204(_result, _user) {
    return false;
  }
  setDataLegacy(data, user) {
    super.setData(data, user);
  }

  // Do not stringify options property since that includes this adapter and will
  // cause a circular reference when Statsig sends diagnostic events and including
  // values is not necessary and makes the result huge
  toJSON() {
    const result = {
      ...this
    };
    delete result._options;
    delete result._inMemoryCache;
    delete result.bootstrapResult;
    return result;
  }
}