// SPDX-License-Identifier: MIT
// Copyright (C) 2018-present iced project and contributors

use crate::iced_constants::IcedConstants;
use crate::iced_error::IcedError;
use core::iter::{ExactSizeIterator, FusedIterator, Iterator};
use core::{fmt, mem};

#[cfg(any(feature = "instr_info", feature = "encoder"))]
pub use crate::memory_size::info::*;

#[cfg(any(feature = "instr_info", feature = "encoder"))]
mod info {
	use crate::iced_constants::IcedConstants;
	use crate::MemorySize;

	#[rustfmt::skip]
	pub(super) static MEMORY_SIZE_INFOS: &[MemorySizeInfo; IcedConstants::MEMORY_SIZE_ENUM_COUNT] = &[
		// GENERATOR-BEGIN: MemorySizeInfoTable
		// ⚠️This was generated by GENERATOR!🦹‍♂️
		MemorySizeInfo { size: 0, element_size: 0, memory_size: MemorySize::Unknown, element_type: MemorySize::Unknown, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 1, element_size: 1, memory_size: MemorySize::UInt8, element_type: MemorySize::UInt8, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::UInt16, element_type: MemorySize::UInt16, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::UInt32, element_type: MemorySize::UInt32, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::UInt52, element_type: MemorySize::UInt52, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::UInt64, element_type: MemorySize::UInt64, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 16, memory_size: MemorySize::UInt128, element_type: MemorySize::UInt128, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 32, memory_size: MemorySize::UInt256, element_type: MemorySize::UInt256, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 64, memory_size: MemorySize::UInt512, element_type: MemorySize::UInt512, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 1, element_size: 1, memory_size: MemorySize::Int8, element_type: MemorySize::Int8, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Int16, element_type: MemorySize::Int16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Int32, element_type: MemorySize::Int32, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Int64, element_type: MemorySize::Int64, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 16, memory_size: MemorySize::Int128, element_type: MemorySize::Int128, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 32, memory_size: MemorySize::Int256, element_type: MemorySize::Int256, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 64, memory_size: MemorySize::Int512, element_type: MemorySize::Int512, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::SegPtr16, element_type: MemorySize::SegPtr16, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 6, element_size: 6, memory_size: MemorySize::SegPtr32, element_type: MemorySize::SegPtr32, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 10, element_size: 10, memory_size: MemorySize::SegPtr64, element_type: MemorySize::SegPtr64, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::WordOffset, element_type: MemorySize::WordOffset, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::DwordOffset, element_type: MemorySize::DwordOffset, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::QwordOffset, element_type: MemorySize::QwordOffset, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 4, element_size: 2, memory_size: MemorySize::Bound16_WordWord, element_type: MemorySize::UInt16, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 4, memory_size: MemorySize::Bound32_DwordDword, element_type: MemorySize::UInt32, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Bnd32, element_type: MemorySize::Bnd32, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 16, memory_size: MemorySize::Bnd64, element_type: MemorySize::Bnd64, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 6, element_size: 6, memory_size: MemorySize::Fword6, element_type: MemorySize::Fword6, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 10, element_size: 10, memory_size: MemorySize::Fword10, element_type: MemorySize::Fword10, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Float16, element_type: MemorySize::Float16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Float32, element_type: MemorySize::Float32, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Float64, element_type: MemorySize::Float64, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 10, element_size: 10, memory_size: MemorySize::Float80, element_type: MemorySize::Float80, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 16, memory_size: MemorySize::Float128, element_type: MemorySize::Float128, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::BFloat16, element_type: MemorySize::BFloat16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 14, element_size: 14, memory_size: MemorySize::FpuEnv14, element_type: MemorySize::FpuEnv14, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 28, element_size: 28, memory_size: MemorySize::FpuEnv28, element_type: MemorySize::FpuEnv28, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 94, element_size: 94, memory_size: MemorySize::FpuState94, element_type: MemorySize::FpuState94, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 108, element_size: 108, memory_size: MemorySize::FpuState108, element_type: MemorySize::FpuState108, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 512, element_size: 512, memory_size: MemorySize::Fxsave_512Byte, element_type: MemorySize::Fxsave_512Byte, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 512, element_size: 512, memory_size: MemorySize::Fxsave64_512Byte, element_type: MemorySize::Fxsave64_512Byte, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 0, element_size: 0, memory_size: MemorySize::Xsave, element_type: MemorySize::Xsave, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 0, element_size: 0, memory_size: MemorySize::Xsave64, element_type: MemorySize::Xsave64, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 10, element_size: 10, memory_size: MemorySize::Bcd, element_type: MemorySize::Bcd, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 64, memory_size: MemorySize::Tilecfg, element_type: MemorySize::Tilecfg, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 0, element_size: 0, memory_size: MemorySize::Tile, element_type: MemorySize::Tile, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 10, element_size: 10, memory_size: MemorySize::SegmentDescSelector, element_type: MemorySize::SegmentDescSelector, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 48, element_size: 48, memory_size: MemorySize::KLHandleAes128, element_type: MemorySize::KLHandleAes128, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 64, memory_size: MemorySize::KLHandleAes256, element_type: MemorySize::KLHandleAes256, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 2, element_size: 1, memory_size: MemorySize::Packed16_UInt8, element_type: MemorySize::UInt8, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 2, element_size: 1, memory_size: MemorySize::Packed16_Int8, element_type: MemorySize::Int8, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 4, element_size: 1, memory_size: MemorySize::Packed32_UInt8, element_type: MemorySize::UInt8, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 4, element_size: 1, memory_size: MemorySize::Packed32_Int8, element_type: MemorySize::Int8, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 4, element_size: 2, memory_size: MemorySize::Packed32_UInt16, element_type: MemorySize::UInt16, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 4, element_size: 2, memory_size: MemorySize::Packed32_Int16, element_type: MemorySize::Int16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 4, element_size: 2, memory_size: MemorySize::Packed32_Float16, element_type: MemorySize::Float16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 4, element_size: 2, memory_size: MemorySize::Packed32_BFloat16, element_type: MemorySize::BFloat16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 1, memory_size: MemorySize::Packed64_UInt8, element_type: MemorySize::UInt8, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 1, memory_size: MemorySize::Packed64_Int8, element_type: MemorySize::Int8, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 2, memory_size: MemorySize::Packed64_UInt16, element_type: MemorySize::UInt16, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 2, memory_size: MemorySize::Packed64_Int16, element_type: MemorySize::Int16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 4, memory_size: MemorySize::Packed64_UInt32, element_type: MemorySize::UInt32, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 4, memory_size: MemorySize::Packed64_Int32, element_type: MemorySize::Int32, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 2, memory_size: MemorySize::Packed64_Float16, element_type: MemorySize::Float16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 8, element_size: 4, memory_size: MemorySize::Packed64_Float32, element_type: MemorySize::Float32, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 1, memory_size: MemorySize::Packed128_UInt8, element_type: MemorySize::UInt8, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 1, memory_size: MemorySize::Packed128_Int8, element_type: MemorySize::Int8, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 2, memory_size: MemorySize::Packed128_UInt16, element_type: MemorySize::UInt16, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 2, memory_size: MemorySize::Packed128_Int16, element_type: MemorySize::Int16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 4, memory_size: MemorySize::Packed128_UInt32, element_type: MemorySize::UInt32, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 4, memory_size: MemorySize::Packed128_Int32, element_type: MemorySize::Int32, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 8, memory_size: MemorySize::Packed128_UInt52, element_type: MemorySize::UInt52, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 8, memory_size: MemorySize::Packed128_UInt64, element_type: MemorySize::UInt64, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 8, memory_size: MemorySize::Packed128_Int64, element_type: MemorySize::Int64, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 2, memory_size: MemorySize::Packed128_Float16, element_type: MemorySize::Float16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 4, memory_size: MemorySize::Packed128_Float32, element_type: MemorySize::Float32, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 8, memory_size: MemorySize::Packed128_Float64, element_type: MemorySize::Float64, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 2, memory_size: MemorySize::Packed128_BFloat16, element_type: MemorySize::BFloat16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 4, memory_size: MemorySize::Packed128_2xFloat16, element_type: MemorySize::Packed32_Float16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 16, element_size: 4, memory_size: MemorySize::Packed128_2xBFloat16, element_type: MemorySize::Packed32_BFloat16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 1, memory_size: MemorySize::Packed256_UInt8, element_type: MemorySize::UInt8, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 1, memory_size: MemorySize::Packed256_Int8, element_type: MemorySize::Int8, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 2, memory_size: MemorySize::Packed256_UInt16, element_type: MemorySize::UInt16, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 2, memory_size: MemorySize::Packed256_Int16, element_type: MemorySize::Int16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 4, memory_size: MemorySize::Packed256_UInt32, element_type: MemorySize::UInt32, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 4, memory_size: MemorySize::Packed256_Int32, element_type: MemorySize::Int32, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 8, memory_size: MemorySize::Packed256_UInt52, element_type: MemorySize::UInt52, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 8, memory_size: MemorySize::Packed256_UInt64, element_type: MemorySize::UInt64, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 8, memory_size: MemorySize::Packed256_Int64, element_type: MemorySize::Int64, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 16, memory_size: MemorySize::Packed256_UInt128, element_type: MemorySize::UInt128, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 16, memory_size: MemorySize::Packed256_Int128, element_type: MemorySize::Int128, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 2, memory_size: MemorySize::Packed256_Float16, element_type: MemorySize::Float16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 4, memory_size: MemorySize::Packed256_Float32, element_type: MemorySize::Float32, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 8, memory_size: MemorySize::Packed256_Float64, element_type: MemorySize::Float64, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 16, memory_size: MemorySize::Packed256_Float128, element_type: MemorySize::Float128, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 2, memory_size: MemorySize::Packed256_BFloat16, element_type: MemorySize::BFloat16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 4, memory_size: MemorySize::Packed256_2xFloat16, element_type: MemorySize::Packed32_Float16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 32, element_size: 4, memory_size: MemorySize::Packed256_2xBFloat16, element_type: MemorySize::Packed32_BFloat16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 1, memory_size: MemorySize::Packed512_UInt8, element_type: MemorySize::UInt8, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 1, memory_size: MemorySize::Packed512_Int8, element_type: MemorySize::Int8, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 2, memory_size: MemorySize::Packed512_UInt16, element_type: MemorySize::UInt16, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 2, memory_size: MemorySize::Packed512_Int16, element_type: MemorySize::Int16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 4, memory_size: MemorySize::Packed512_UInt32, element_type: MemorySize::UInt32, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 4, memory_size: MemorySize::Packed512_Int32, element_type: MemorySize::Int32, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 8, memory_size: MemorySize::Packed512_UInt52, element_type: MemorySize::UInt52, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 8, memory_size: MemorySize::Packed512_UInt64, element_type: MemorySize::UInt64, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 8, memory_size: MemorySize::Packed512_Int64, element_type: MemorySize::Int64, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 16, memory_size: MemorySize::Packed512_UInt128, element_type: MemorySize::UInt128, is_signed: false, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 2, memory_size: MemorySize::Packed512_Float16, element_type: MemorySize::Float16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 4, memory_size: MemorySize::Packed512_Float32, element_type: MemorySize::Float32, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 8, memory_size: MemorySize::Packed512_Float64, element_type: MemorySize::Float64, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 4, memory_size: MemorySize::Packed512_2xFloat16, element_type: MemorySize::Packed32_Float16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 64, element_size: 4, memory_size: MemorySize::Packed512_2xBFloat16, element_type: MemorySize::Packed32_BFloat16, is_signed: true, is_broadcast: false },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Broadcast32_Float16, element_type: MemorySize::Float16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast64_UInt32, element_type: MemorySize::UInt32, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast64_Int32, element_type: MemorySize::Int32, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Broadcast64_Float16, element_type: MemorySize::Float16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast64_Float32, element_type: MemorySize::Float32, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Broadcast128_Int16, element_type: MemorySize::Int16, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Broadcast128_UInt16, element_type: MemorySize::UInt16, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast128_UInt32, element_type: MemorySize::UInt32, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast128_Int32, element_type: MemorySize::Int32, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast128_UInt52, element_type: MemorySize::UInt52, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast128_UInt64, element_type: MemorySize::UInt64, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast128_Int64, element_type: MemorySize::Int64, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Broadcast128_Float16, element_type: MemorySize::Float16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast128_Float32, element_type: MemorySize::Float32, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast128_Float64, element_type: MemorySize::Float64, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast128_2xInt16, element_type: MemorySize::Packed32_Int16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast128_2xInt32, element_type: MemorySize::Packed64_Int32, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast128_2xUInt32, element_type: MemorySize::Packed64_UInt32, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast128_2xFloat16, element_type: MemorySize::Packed32_Float16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast128_2xBFloat16, element_type: MemorySize::Packed32_BFloat16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Broadcast256_Int16, element_type: MemorySize::Int16, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Broadcast256_UInt16, element_type: MemorySize::UInt16, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast256_UInt32, element_type: MemorySize::UInt32, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast256_Int32, element_type: MemorySize::Int32, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast256_UInt52, element_type: MemorySize::UInt52, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast256_UInt64, element_type: MemorySize::UInt64, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast256_Int64, element_type: MemorySize::Int64, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Broadcast256_Float16, element_type: MemorySize::Float16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast256_Float32, element_type: MemorySize::Float32, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast256_Float64, element_type: MemorySize::Float64, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast256_2xInt16, element_type: MemorySize::Packed32_Int16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast256_2xInt32, element_type: MemorySize::Packed64_Int32, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast256_2xUInt32, element_type: MemorySize::Packed64_UInt32, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast256_2xFloat16, element_type: MemorySize::Packed32_Float16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast256_2xBFloat16, element_type: MemorySize::Packed32_BFloat16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Broadcast512_Int16, element_type: MemorySize::Int16, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Broadcast512_UInt16, element_type: MemorySize::UInt16, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast512_UInt32, element_type: MemorySize::UInt32, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast512_Int32, element_type: MemorySize::Int32, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast512_UInt52, element_type: MemorySize::UInt52, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast512_UInt64, element_type: MemorySize::UInt64, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast512_Int64, element_type: MemorySize::Int64, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 2, element_size: 2, memory_size: MemorySize::Broadcast512_Float16, element_type: MemorySize::Float16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast512_Float32, element_type: MemorySize::Float32, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast512_Float64, element_type: MemorySize::Float64, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast512_2xFloat16, element_type: MemorySize::Packed32_Float16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast512_2xInt16, element_type: MemorySize::Packed32_Int16, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast512_2xUInt32, element_type: MemorySize::Packed64_UInt32, is_signed: false, is_broadcast: true },
		MemorySizeInfo { size: 8, element_size: 8, memory_size: MemorySize::Broadcast512_2xInt32, element_type: MemorySize::Packed64_Int32, is_signed: true, is_broadcast: true },
		MemorySizeInfo { size: 4, element_size: 4, memory_size: MemorySize::Broadcast512_2xBFloat16, element_type: MemorySize::Packed32_BFloat16, is_signed: true, is_broadcast: true },
		// GENERATOR-END: MemorySizeInfoTable
	];

	/// [`MemorySize`] information
	///
	/// [`MemorySize`]: enum.MemorySize.html
	#[derive(Debug, Copy, Clone, Eq, PartialEq, Hash)]
	pub struct MemorySizeInfo {
		size: u16,
		element_size: u16,
		memory_size: MemorySize,
		element_type: MemorySize,
		// Use flags if more booleans are needed
		is_signed: bool,
		is_broadcast: bool,
	}

	impl MemorySizeInfo {
		/// Gets the [`MemorySize`] value
		///
		/// [`MemorySize`]: enum.MemorySize.html
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = MemorySize::Packed256_UInt16.info();
		/// assert_eq!(info.memory_size(), MemorySize::Packed256_UInt16);
		/// ```
		#[must_use]
		#[inline]
		pub const fn memory_size(&self) -> MemorySize {
			self.memory_size
		}

		/// Gets the size in bytes of the memory location or 0 if it's not accessed or unknown
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = MemorySize::UInt32.info();
		/// assert_eq!(info.size(), 4);
		/// let info = MemorySize::Packed256_UInt16.info();
		/// assert_eq!(info.size(), 32);
		/// let info = MemorySize::Broadcast512_UInt64.info();
		/// assert_eq!(info.size(), 8);
		/// ```
		#[must_use]
		#[inline]
		pub const fn size(&self) -> usize {
			self.size as usize
		}

		/// Gets the size in bytes of the packed element. If it's not a packed data type, it's equal to [`size()`].
		///
		/// [`size()`]: #method.size
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = MemorySize::UInt32.info();
		/// assert_eq!(info.element_size(), 4);
		/// let info = MemorySize::Packed256_UInt16.info();
		/// assert_eq!(info.element_size(), 2);
		/// let info = MemorySize::Broadcast512_UInt64.info();
		/// assert_eq!(info.element_size(), 8);
		/// ```
		#[must_use]
		#[inline]
		pub const fn element_size(&self) -> usize {
			self.element_size as usize
		}

		/// Gets the element type if it's packed data or the type itself if it's not packed data
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = MemorySize::UInt32.info();
		/// assert_eq!(info.element_type(), MemorySize::UInt32);
		/// let info = MemorySize::Packed256_UInt16.info();
		/// assert_eq!(info.element_type(), MemorySize::UInt16);
		/// let info = MemorySize::Broadcast512_UInt64.info();
		/// assert_eq!(info.element_type(), MemorySize::UInt64);
		/// ```
		#[must_use]
		#[inline]
		pub const fn element_type(&self) -> MemorySize {
			self.element_type
		}

		/// Gets the element type if it's packed data or the type itself if it's not packed data
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = MemorySize::UInt32.info().element_type_info();
		/// assert_eq!(info.memory_size(), MemorySize::UInt32);
		/// let info = MemorySize::Packed256_UInt16.info().element_type_info();
		/// assert_eq!(info.memory_size(), MemorySize::UInt16);
		/// let info = MemorySize::Broadcast512_UInt64.info().element_type_info();
		/// assert_eq!(info.memory_size(), MemorySize::UInt64);
		/// ```
		#[must_use]
		#[inline]
		pub fn element_type_info(&self) -> &'static Self {
			self.element_type().info()
		}

		/// `true` if it's signed data (signed integer or a floating point value)
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = MemorySize::UInt32.info();
		/// assert!(!info.is_signed());
		/// let info = MemorySize::Int32.info();
		/// assert!(info.is_signed());
		/// let info = MemorySize::Float64.info();
		/// assert!(info.is_signed());
		/// ```
		#[must_use]
		#[inline]
		pub const fn is_signed(&self) -> bool {
			self.is_signed
		}

		/// `true` if it's a broadcast memory type
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = MemorySize::UInt32.info();
		/// assert!(!info.is_broadcast());
		/// let info = MemorySize::Packed256_UInt16.info();
		/// assert!(!info.is_broadcast());
		/// let info = MemorySize::Broadcast512_UInt64.info();
		/// assert!(info.is_broadcast());
		/// ```
		#[must_use]
		#[inline]
		pub const fn is_broadcast(&self) -> bool {
			self.is_broadcast
		}

		/// `true` if this is a packed data type, eg. [`MemorySize::Packed128_Float32`]. See also [`element_count()`]
		///
		/// [`MemorySize::Packed128_Float32`]: #variant.Packed128_Float32
		/// [`element_count()`]: #method.element_count
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = MemorySize::UInt32.info();
		/// assert!(!info.is_packed());
		/// let info = MemorySize::Packed256_UInt16.info();
		/// assert!(info.is_packed());
		/// let info = MemorySize::Broadcast512_UInt64.info();
		/// assert!(!info.is_packed());
		/// ```
		#[must_use]
		#[inline]
		pub const fn is_packed(&self) -> bool {
			self.element_size < self.size
		}

		/// Gets the number of elements in the packed data type or `1` if it's not packed data ([`is_packed()`])
		///
		/// [`is_packed()`]: #method.is_packed
		///
		/// # Examples
		///
		/// ```
		/// use iced_x86::*;
		/// let info = MemorySize::UInt32.info();
		/// assert_eq!(info.element_count(), 1);
		/// let info = MemorySize::Packed256_UInt16.info();
		/// assert_eq!(info.element_count(), 16);
		/// let info = MemorySize::Broadcast512_UInt64.info();
		/// assert_eq!(info.element_count(), 1);
		/// ```
		#[must_use]
		#[inline]
		pub const fn element_count(&self) -> usize {
			// element_size can be 0 so we don't divide by it if es == s
			if self.element_size == self.size {
				1
			} else {
				self.size as usize / self.element_size as usize
			}
		}
	}
}

// GENERATOR-BEGIN: MemorySize
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// Size of a memory reference
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
#[cfg_attr(not(feature = "exhaustive_enums"), non_exhaustive)]
#[allow(non_camel_case_types)]
pub enum MemorySize {
	/// Unknown size or the instruction doesn't reference any memory (eg. `LEA`)
	Unknown = 0,
	/// Memory location contains a `u8`
	UInt8 = 1,
	/// Memory location contains a `u16`
	UInt16 = 2,
	/// Memory location contains a `u32`
	UInt32 = 3,
	/// Memory location contains a `u52`
	UInt52 = 4,
	/// Memory location contains a `u64`
	UInt64 = 5,
	/// Memory location contains a `u128`
	UInt128 = 6,
	/// Memory location contains a `u256`
	UInt256 = 7,
	/// Memory location contains a `u512`
	UInt512 = 8,
	/// Memory location contains a `i8`
	Int8 = 9,
	/// Memory location contains a `i16`
	Int16 = 10,
	/// Memory location contains a `i32`
	Int32 = 11,
	/// Memory location contains a `i64`
	Int64 = 12,
	/// Memory location contains a `i128`
	Int128 = 13,
	/// Memory location contains a `i256`
	Int256 = 14,
	/// Memory location contains a `i512`
	Int512 = 15,
	/// Memory location contains a seg:ptr pair, `u16` (offset) + `u16` (segment/selector)
	SegPtr16 = 16,
	/// Memory location contains a seg:ptr pair, `u32` (offset) + `u16` (segment/selector)
	SegPtr32 = 17,
	/// Memory location contains a seg:ptr pair, `u64` (offset) + `u16` (segment/selector)
	SegPtr64 = 18,
	/// Memory location contains a 16-bit offset (`JMP/CALL WORD PTR [mem]`)
	WordOffset = 19,
	/// Memory location contains a 32-bit offset (`JMP/CALL DWORD PTR [mem]`)
	DwordOffset = 20,
	/// Memory location contains a 64-bit offset (`JMP/CALL QWORD PTR [mem]`)
	QwordOffset = 21,
	/// Memory location contains two `u16`s (16-bit `BOUND`)
	Bound16_WordWord = 22,
	/// Memory location contains two `u32`s (32-bit `BOUND`)
	Bound32_DwordDword = 23,
	/// 32-bit `BNDMOV`, 2 x `u32`
	Bnd32 = 24,
	/// 64-bit `BNDMOV`, 2 x `u64`
	Bnd64 = 25,
	/// Memory location contains a 16-bit limit and a 32-bit address (eg. `LGDTW`, `LGDTD`)
	Fword6 = 26,
	/// Memory location contains a 16-bit limit and a 64-bit address (eg. `LGDTQ`)
	Fword10 = 27,
	/// Memory location contains a `f16`
	Float16 = 28,
	/// Memory location contains a `f32`
	Float32 = 29,
	/// Memory location contains a `f64`
	Float64 = 30,
	/// Memory location contains a `f80`
	Float80 = 31,
	/// Memory location contains a `f128`
	Float128 = 32,
	/// Memory location contains a `bfloat16`
	BFloat16 = 33,
	/// Memory location contains a 14-byte FPU environment (16-bit `FLDENV`/`FSTENV`)
	FpuEnv14 = 34,
	/// Memory location contains a 28-byte FPU environment (32/64-bit `FLDENV`/`FSTENV`)
	FpuEnv28 = 35,
	/// Memory location contains a 94-byte FPU environment (16-bit `FSAVE`/`FRSTOR`)
	FpuState94 = 36,
	/// Memory location contains a 108-byte FPU environment (32/64-bit `FSAVE`/`FRSTOR`)
	FpuState108 = 37,
	/// Memory location contains 512-bytes of `FXSAVE`/`FXRSTOR` data
	Fxsave_512Byte = 38,
	/// Memory location contains 512-bytes of `FXSAVE64`/`FXRSTOR64` data
	Fxsave64_512Byte = 39,
	/// 32-bit `XSAVE` area
	Xsave = 40,
	/// 64-bit `XSAVE` area
	Xsave64 = 41,
	/// Memory location contains a 10-byte `bcd` value (`FBLD`/`FBSTP`)
	Bcd = 42,
	/// 64-bit location: TILECFG (`LDTILECFG`/`STTILECFG`)
	Tilecfg = 43,
	/// Tile data
	Tile = 44,
	/// 80-bit segment descriptor and selector: 0-7 = descriptor, 8-9 = selector
	SegmentDescSelector = 45,
	/// 384-bit AES 128 handle (Key Locker)
	KLHandleAes128 = 46,
	/// 512-bit AES 256 handle (Key Locker)
	KLHandleAes256 = 47,
	/// 16-bit location: 2 x `u8`
	Packed16_UInt8 = 48,
	/// 16-bit location: 2 x `i8`
	Packed16_Int8 = 49,
	/// 32-bit location: 4 x `u8`
	Packed32_UInt8 = 50,
	/// 32-bit location: 4 x `i8`
	Packed32_Int8 = 51,
	/// 32-bit location: 2 x `u16`
	Packed32_UInt16 = 52,
	/// 32-bit location: 2 x `i16`
	Packed32_Int16 = 53,
	/// 32-bit location: 2 x `f16`
	Packed32_Float16 = 54,
	/// 32-bit location: 2 x `bfloat16`
	Packed32_BFloat16 = 55,
	/// 64-bit location: 8 x `u8`
	Packed64_UInt8 = 56,
	/// 64-bit location: 8 x `i8`
	Packed64_Int8 = 57,
	/// 64-bit location: 4 x `u16`
	Packed64_UInt16 = 58,
	/// 64-bit location: 4 x `i16`
	Packed64_Int16 = 59,
	/// 64-bit location: 2 x `u32`
	Packed64_UInt32 = 60,
	/// 64-bit location: 2 x `i32`
	Packed64_Int32 = 61,
	/// 64-bit location: 4 x `f16`
	Packed64_Float16 = 62,
	/// 64-bit location: 2 x `f32`
	Packed64_Float32 = 63,
	/// 128-bit location: 16 x `u8`
	Packed128_UInt8 = 64,
	/// 128-bit location: 16 x `i8`
	Packed128_Int8 = 65,
	/// 128-bit location: 8 x `u16`
	Packed128_UInt16 = 66,
	/// 128-bit location: 8 x `i16`
	Packed128_Int16 = 67,
	/// 128-bit location: 4 x `u32`
	Packed128_UInt32 = 68,
	/// 128-bit location: 4 x `i32`
	Packed128_Int32 = 69,
	/// 128-bit location: 2 x `u52`
	Packed128_UInt52 = 70,
	/// 128-bit location: 2 x `u64`
	Packed128_UInt64 = 71,
	/// 128-bit location: 2 x `i64`
	Packed128_Int64 = 72,
	/// 128-bit location: 8 x `f16`
	Packed128_Float16 = 73,
	/// 128-bit location: 4 x `f32`
	Packed128_Float32 = 74,
	/// 128-bit location: 2 x `f64`
	Packed128_Float64 = 75,
	/// 128-bit location: 8 x `bfloat16`
	Packed128_BFloat16 = 76,
	/// 128-bit location: 4 x (2 x `f16`)
	Packed128_2xFloat16 = 77,
	/// 128-bit location: 4 x (2 x `bfloat16`)
	Packed128_2xBFloat16 = 78,
	/// 256-bit location: 32 x `u8`
	Packed256_UInt8 = 79,
	/// 256-bit location: 32 x `i8`
	Packed256_Int8 = 80,
	/// 256-bit location: 16 x `u16`
	Packed256_UInt16 = 81,
	/// 256-bit location: 16 x `i16`
	Packed256_Int16 = 82,
	/// 256-bit location: 8 x `u32`
	Packed256_UInt32 = 83,
	/// 256-bit location: 8 x `i32`
	Packed256_Int32 = 84,
	/// 256-bit location: 4 x `u52`
	Packed256_UInt52 = 85,
	/// 256-bit location: 4 x `u64`
	Packed256_UInt64 = 86,
	/// 256-bit location: 4 x `i64`
	Packed256_Int64 = 87,
	/// 256-bit location: 2 x `u128`
	Packed256_UInt128 = 88,
	/// 256-bit location: 2 x `i128`
	Packed256_Int128 = 89,
	/// 256-bit location: 16 x `f16`
	Packed256_Float16 = 90,
	/// 256-bit location: 8 x `f32`
	Packed256_Float32 = 91,
	/// 256-bit location: 4 x `f64`
	Packed256_Float64 = 92,
	/// 256-bit location: 2 x `f128`
	Packed256_Float128 = 93,
	/// 256-bit location: 16 x `bfloat16`
	Packed256_BFloat16 = 94,
	/// 256-bit location: 8 x (2 x `f16`)
	Packed256_2xFloat16 = 95,
	/// 256-bit location: 8 x (2 x `bfloat16`)
	Packed256_2xBFloat16 = 96,
	/// 512-bit location: 64 x `u8`
	Packed512_UInt8 = 97,
	/// 512-bit location: 64 x `i8`
	Packed512_Int8 = 98,
	/// 512-bit location: 32 x `u16`
	Packed512_UInt16 = 99,
	/// 512-bit location: 32 x `i16`
	Packed512_Int16 = 100,
	/// 512-bit location: 16 x `u32`
	Packed512_UInt32 = 101,
	/// 512-bit location: 16 x `i32`
	Packed512_Int32 = 102,
	/// 512-bit location: 8 x `u52`
	Packed512_UInt52 = 103,
	/// 512-bit location: 8 x `u64`
	Packed512_UInt64 = 104,
	/// 512-bit location: 8 x `i64`
	Packed512_Int64 = 105,
	/// 256-bit location: 4 x `u128`
	Packed512_UInt128 = 106,
	/// 512-bit location: 32 x `f16`
	Packed512_Float16 = 107,
	/// 512-bit location: 16 x `f32`
	Packed512_Float32 = 108,
	/// 512-bit location: 8 x `f64`
	Packed512_Float64 = 109,
	/// 512-bit location: 16 x (2 x `f16`)
	Packed512_2xFloat16 = 110,
	/// 512-bit location: 16 x (2 x `bfloat16`)
	Packed512_2xBFloat16 = 111,
	/// Broadcast `f16` to 32-bits
	Broadcast32_Float16 = 112,
	/// Broadcast `u32` to 64-bits
	Broadcast64_UInt32 = 113,
	/// Broadcast `i32` to 64-bits
	Broadcast64_Int32 = 114,
	/// Broadcast `f16` to 64-bits
	Broadcast64_Float16 = 115,
	/// Broadcast `f32` to 64-bits
	Broadcast64_Float32 = 116,
	/// Broadcast `i16` to 128-bits
	Broadcast128_Int16 = 117,
	/// Broadcast `u16` to 128-bits
	Broadcast128_UInt16 = 118,
	/// Broadcast `u32` to 128-bits
	Broadcast128_UInt32 = 119,
	/// Broadcast `i32` to 128-bits
	Broadcast128_Int32 = 120,
	/// Broadcast `u52` to 128-bits
	Broadcast128_UInt52 = 121,
	/// Broadcast `u64` to 128-bits
	Broadcast128_UInt64 = 122,
	/// Broadcast `i64` to 128-bits
	Broadcast128_Int64 = 123,
	/// Broadcast `f16` to 128-bits
	Broadcast128_Float16 = 124,
	/// Broadcast `f32` to 128-bits
	Broadcast128_Float32 = 125,
	/// Broadcast `f64` to 128-bits
	Broadcast128_Float64 = 126,
	/// Broadcast 2 x `i16` to 128-bits
	Broadcast128_2xInt16 = 127,
	/// Broadcast 2 x `i32` to 128-bits
	Broadcast128_2xInt32 = 128,
	/// Broadcast 2 x `u32` to 128-bits
	Broadcast128_2xUInt32 = 129,
	/// Broadcast 2 x `f16` to 128-bits
	Broadcast128_2xFloat16 = 130,
	/// Broadcast 2 x `bfloat16` to 128-bits
	Broadcast128_2xBFloat16 = 131,
	/// Broadcast `i16` to 256-bits
	Broadcast256_Int16 = 132,
	/// Broadcast `u16` to 256-bits
	Broadcast256_UInt16 = 133,
	/// Broadcast `u32` to 256-bits
	Broadcast256_UInt32 = 134,
	/// Broadcast `i32` to 256-bits
	Broadcast256_Int32 = 135,
	/// Broadcast `u52` to 256-bits
	Broadcast256_UInt52 = 136,
	/// Broadcast `u64` to 256-bits
	Broadcast256_UInt64 = 137,
	/// Broadcast `i64` to 256-bits
	Broadcast256_Int64 = 138,
	/// Broadcast `f16` to 256-bits
	Broadcast256_Float16 = 139,
	/// Broadcast `f32` to 256-bits
	Broadcast256_Float32 = 140,
	/// Broadcast `f64` to 256-bits
	Broadcast256_Float64 = 141,
	/// Broadcast 2 x `i16` to 256-bits
	Broadcast256_2xInt16 = 142,
	/// Broadcast 2 x `i32` to 256-bits
	Broadcast256_2xInt32 = 143,
	/// Broadcast 2 x `u32` to 256-bits
	Broadcast256_2xUInt32 = 144,
	/// Broadcast 2 x `f16` to 256-bits
	Broadcast256_2xFloat16 = 145,
	/// Broadcast 2 x `bfloat16` to 256-bits
	Broadcast256_2xBFloat16 = 146,
	/// Broadcast `i16` to 512-bits
	Broadcast512_Int16 = 147,
	/// Broadcast `u16` to 512-bits
	Broadcast512_UInt16 = 148,
	/// Broadcast `u32` to 512-bits
	Broadcast512_UInt32 = 149,
	/// Broadcast `i32` to 512-bits
	Broadcast512_Int32 = 150,
	/// Broadcast `u52` to 512-bits
	Broadcast512_UInt52 = 151,
	/// Broadcast `u64` to 512-bits
	Broadcast512_UInt64 = 152,
	/// Broadcast `i64` to 512-bits
	Broadcast512_Int64 = 153,
	/// Broadcast `f16` to 512-bits
	Broadcast512_Float16 = 154,
	/// Broadcast `f32` to 512-bits
	Broadcast512_Float32 = 155,
	/// Broadcast `f64` to 512-bits
	Broadcast512_Float64 = 156,
	/// Broadcast 2 x `f16` to 512-bits
	Broadcast512_2xFloat16 = 157,
	/// Broadcast 2 x `i16` to 512-bits
	Broadcast512_2xInt16 = 158,
	/// Broadcast 2 x `u32` to 512-bits
	Broadcast512_2xUInt32 = 159,
	/// Broadcast 2 x `i32` to 512-bits
	Broadcast512_2xInt32 = 160,
	/// Broadcast 2 x `bfloat16` to 512-bits
	Broadcast512_2xBFloat16 = 161,
}
#[rustfmt::skip]
static GEN_DEBUG_MEMORY_SIZE: [&str; 162] = [
	"Unknown",
	"UInt8",
	"UInt16",
	"UInt32",
	"UInt52",
	"UInt64",
	"UInt128",
	"UInt256",
	"UInt512",
	"Int8",
	"Int16",
	"Int32",
	"Int64",
	"Int128",
	"Int256",
	"Int512",
	"SegPtr16",
	"SegPtr32",
	"SegPtr64",
	"WordOffset",
	"DwordOffset",
	"QwordOffset",
	"Bound16_WordWord",
	"Bound32_DwordDword",
	"Bnd32",
	"Bnd64",
	"Fword6",
	"Fword10",
	"Float16",
	"Float32",
	"Float64",
	"Float80",
	"Float128",
	"BFloat16",
	"FpuEnv14",
	"FpuEnv28",
	"FpuState94",
	"FpuState108",
	"Fxsave_512Byte",
	"Fxsave64_512Byte",
	"Xsave",
	"Xsave64",
	"Bcd",
	"Tilecfg",
	"Tile",
	"SegmentDescSelector",
	"KLHandleAes128",
	"KLHandleAes256",
	"Packed16_UInt8",
	"Packed16_Int8",
	"Packed32_UInt8",
	"Packed32_Int8",
	"Packed32_UInt16",
	"Packed32_Int16",
	"Packed32_Float16",
	"Packed32_BFloat16",
	"Packed64_UInt8",
	"Packed64_Int8",
	"Packed64_UInt16",
	"Packed64_Int16",
	"Packed64_UInt32",
	"Packed64_Int32",
	"Packed64_Float16",
	"Packed64_Float32",
	"Packed128_UInt8",
	"Packed128_Int8",
	"Packed128_UInt16",
	"Packed128_Int16",
	"Packed128_UInt32",
	"Packed128_Int32",
	"Packed128_UInt52",
	"Packed128_UInt64",
	"Packed128_Int64",
	"Packed128_Float16",
	"Packed128_Float32",
	"Packed128_Float64",
	"Packed128_BFloat16",
	"Packed128_2xFloat16",
	"Packed128_2xBFloat16",
	"Packed256_UInt8",
	"Packed256_Int8",
	"Packed256_UInt16",
	"Packed256_Int16",
	"Packed256_UInt32",
	"Packed256_Int32",
	"Packed256_UInt52",
	"Packed256_UInt64",
	"Packed256_Int64",
	"Packed256_UInt128",
	"Packed256_Int128",
	"Packed256_Float16",
	"Packed256_Float32",
	"Packed256_Float64",
	"Packed256_Float128",
	"Packed256_BFloat16",
	"Packed256_2xFloat16",
	"Packed256_2xBFloat16",
	"Packed512_UInt8",
	"Packed512_Int8",
	"Packed512_UInt16",
	"Packed512_Int16",
	"Packed512_UInt32",
	"Packed512_Int32",
	"Packed512_UInt52",
	"Packed512_UInt64",
	"Packed512_Int64",
	"Packed512_UInt128",
	"Packed512_Float16",
	"Packed512_Float32",
	"Packed512_Float64",
	"Packed512_2xFloat16",
	"Packed512_2xBFloat16",
	"Broadcast32_Float16",
	"Broadcast64_UInt32",
	"Broadcast64_Int32",
	"Broadcast64_Float16",
	"Broadcast64_Float32",
	"Broadcast128_Int16",
	"Broadcast128_UInt16",
	"Broadcast128_UInt32",
	"Broadcast128_Int32",
	"Broadcast128_UInt52",
	"Broadcast128_UInt64",
	"Broadcast128_Int64",
	"Broadcast128_Float16",
	"Broadcast128_Float32",
	"Broadcast128_Float64",
	"Broadcast128_2xInt16",
	"Broadcast128_2xInt32",
	"Broadcast128_2xUInt32",
	"Broadcast128_2xFloat16",
	"Broadcast128_2xBFloat16",
	"Broadcast256_Int16",
	"Broadcast256_UInt16",
	"Broadcast256_UInt32",
	"Broadcast256_Int32",
	"Broadcast256_UInt52",
	"Broadcast256_UInt64",
	"Broadcast256_Int64",
	"Broadcast256_Float16",
	"Broadcast256_Float32",
	"Broadcast256_Float64",
	"Broadcast256_2xInt16",
	"Broadcast256_2xInt32",
	"Broadcast256_2xUInt32",
	"Broadcast256_2xFloat16",
	"Broadcast256_2xBFloat16",
	"Broadcast512_Int16",
	"Broadcast512_UInt16",
	"Broadcast512_UInt32",
	"Broadcast512_Int32",
	"Broadcast512_UInt52",
	"Broadcast512_UInt64",
	"Broadcast512_Int64",
	"Broadcast512_Float16",
	"Broadcast512_Float32",
	"Broadcast512_Float64",
	"Broadcast512_2xFloat16",
	"Broadcast512_2xInt16",
	"Broadcast512_2xUInt32",
	"Broadcast512_2xInt32",
	"Broadcast512_2xBFloat16",
];
impl fmt::Debug for MemorySize {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_MEMORY_SIZE[*self as usize])
	}
}
impl Default for MemorySize {
	#[must_use]
	#[inline]
	fn default() -> Self {
		MemorySize::Unknown
	}
}
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type MemorySizeUnderlyingType = u8;
#[rustfmt::skip]
impl MemorySize {
	/// Iterates over all `MemorySize` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = MemorySize> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::MEMORY_SIZE_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, MemorySize>(x as u8) })
	}
}
#[test]
#[rustfmt::skip]
fn test_memorysize_values() {
	let mut iter = MemorySize::values();
	assert_eq!(iter.size_hint(), (IcedConstants::MEMORY_SIZE_ENUM_COUNT, Some(IcedConstants::MEMORY_SIZE_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::MEMORY_SIZE_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::MEMORY_SIZE_ENUM_COUNT - 1, Some(IcedConstants::MEMORY_SIZE_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::MEMORY_SIZE_ENUM_COUNT - 1);

	let values: Vec<MemorySize> = MemorySize::values().collect();
	assert_eq!(values.len(), IcedConstants::MEMORY_SIZE_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<MemorySize> = MemorySize::values().collect();
	let mut values2: Vec<MemorySize> = MemorySize::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[rustfmt::skip]
impl TryFrom<usize> for MemorySize {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::MEMORY_SIZE_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid MemorySize value"))
		}
	}
}
#[test]
#[rustfmt::skip]
fn test_memorysize_try_from_usize() {
	for value in MemorySize::values() {
		let converted = <MemorySize as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<MemorySize as TryFrom<usize>>::try_from(IcedConstants::MEMORY_SIZE_ENUM_COUNT).is_err());
	assert!(<MemorySize as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
#[cfg(feature = "serde")]
#[rustfmt::skip]
#[allow(clippy::zero_sized_map_values)]
const _: () = {
	use core::marker::PhantomData;
	use serde::de;
	use serde::{Deserialize, Deserializer, Serialize, Serializer};
	type EnumType = MemorySize;
	impl Serialize for EnumType {
		#[inline]
		fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
		where
			S: Serializer,
		{
			serializer.serialize_u8(*self as u8)
		}
	}
	impl<'de> Deserialize<'de> for EnumType {
		#[inline]
		fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
		where
			D: Deserializer<'de>,
		{
			struct Visitor<'de> {
				marker: PhantomData<EnumType>,
				lifetime: PhantomData<&'de ()>,
			}
			impl<'de> de::Visitor<'de> for Visitor<'de> {
				type Value = EnumType;
				#[inline]
				fn expecting(&self, formatter: &mut fmt::Formatter<'_>) -> fmt::Result {
					formatter.write_str("enum MemorySize")
				}
				#[inline]
				fn visit_u64<E>(self, v: u64) -> Result<Self::Value, E>
				where
					E: de::Error,
				{
					if let Ok(v) = <usize as TryFrom<_>>::try_from(v) {
						if let Ok(value) = <EnumType as TryFrom<_>>::try_from(v) {
							return Ok(value);
						}
					}
					Err(de::Error::invalid_value(de::Unexpected::Unsigned(v), &"a valid MemorySize variant value"))
				}
			}
			deserializer.deserialize_u8(Visitor { marker: PhantomData::<EnumType>, lifetime: PhantomData })
		}
	}
};
// GENERATOR-END: MemorySize

#[cfg(any(feature = "instr_info", feature = "encoder"))]
impl MemorySize {
	/// Gets the memory size info
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// let info = MemorySize::Packed256_UInt16.info();
	/// assert_eq!(info.size(), 32);
	/// ```
	#[must_use]
	#[inline]
	pub fn info(self) -> &'static MemorySizeInfo {
		&MEMORY_SIZE_INFOS[self as usize]
	}

	/// Gets the size in bytes of the memory location or 0 if it's not accessed by the instruction or unknown or variable sized
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert_eq!(MemorySize::UInt32.size(), 4);
	/// assert_eq!(MemorySize::Packed256_UInt16.size(), 32);
	/// assert_eq!(MemorySize::Broadcast512_UInt64.size(), 8);
	/// ```
	#[must_use]
	#[inline]
	pub fn size(self) -> usize {
		self.info().size()
	}

	/// Gets the size in bytes of the packed element. If it's not a packed data type, it's equal to [`size()`].
	///
	/// [`size()`]: #method.size
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert_eq!(MemorySize::UInt32.element_size(), 4);
	/// assert_eq!(MemorySize::Packed256_UInt16.element_size(), 2);
	/// assert_eq!(MemorySize::Broadcast512_UInt64.element_size(), 8);
	/// ```
	#[must_use]
	#[inline]
	pub fn element_size(self) -> usize {
		self.info().element_size()
	}

	/// Gets the element type if it's packed data or `self` if it's not packed data
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert_eq!(MemorySize::UInt32.element_type(), MemorySize::UInt32);
	/// assert_eq!(MemorySize::Packed256_UInt16.element_type(), MemorySize::UInt16);
	/// assert_eq!(MemorySize::Broadcast512_UInt64.element_type(), MemorySize::UInt64);
	/// ```
	#[must_use]
	#[inline]
	pub fn element_type(self) -> Self {
		self.info().element_type()
	}

	/// Gets the element type info if it's packed data or `self` if it's not packed data
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert_eq!(MemorySize::UInt32.element_type_info().memory_size(), MemorySize::UInt32);
	/// assert_eq!(MemorySize::Packed256_UInt16.element_type_info().memory_size(), MemorySize::UInt16);
	/// assert_eq!(MemorySize::Broadcast512_UInt64.element_type_info().memory_size(), MemorySize::UInt64);
	/// ```
	#[must_use]
	#[inline]
	pub fn element_type_info(self) -> &'static MemorySizeInfo {
		self.info().element_type().info()
	}

	/// `true` if it's signed data (signed integer or a floating point value)
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!MemorySize::UInt32.is_signed());
	/// assert!(MemorySize::Int32.is_signed());
	/// assert!(MemorySize::Float64.is_signed());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_signed(self) -> bool {
		self.info().is_signed()
	}

	/// `true` if this is a packed data type, eg. [`MemorySize::Packed128_Float32`]
	///
	/// [`MemorySize::Packed128_Float32`]: #variant.Packed128_Float32
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!MemorySize::UInt32.is_packed());
	/// assert!(MemorySize::Packed256_UInt16.is_packed());
	/// assert!(!MemorySize::Broadcast512_UInt64.is_packed());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_packed(self) -> bool {
		self.info().is_packed()
	}

	/// Gets the number of elements in the packed data type or `1` if it's not packed data ([`is_packed()`])
	///
	/// [`is_packed()`]: #method.is_packed
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert_eq!(MemorySize::UInt32.element_count(), 1);
	/// assert_eq!(MemorySize::Packed256_UInt16.element_count(), 16);
	/// assert_eq!(MemorySize::Broadcast512_UInt64.element_count(), 1);
	/// ```
	#[must_use]
	#[inline]
	pub fn element_count(self) -> usize {
		self.info().element_count()
	}
}

#[cfg(any(feature = "instr_info", feature = "encoder", feature = "fast_fmt"))]
impl MemorySize {
	/// Checks if it is a broadcast memory type
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	/// assert!(!MemorySize::Packed64_Float16.is_broadcast());
	/// assert!(MemorySize::Broadcast512_UInt64.is_broadcast());
	/// ```
	#[must_use]
	#[inline]
	pub fn is_broadcast(self) -> bool {
		self >= IcedConstants::FIRST_BROADCAST_MEMORY_SIZE
	}
}
