// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"
#include "Rivet/Projections/ChargedFinalState.hh"
#include "Rivet/Projections/Thrust.hh"
#include "Rivet/Projections/Sphericity.hh"

namespace Rivet {


  /// @brief Event shapes at Upsilon(1S)
  class ARGUS_1986_I227324 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(ARGUS_1986_I227324);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projections
      declare(UnstableParticles(), "UFS");
      declare(ChargedFinalState(), "CFS");
      const FinalState fs;
      declare(Thrust(fs)    ,"Thrust");
      declare(Sphericity(fs),"Sphericity");
      // histograms
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal, 1e-3))  _sqs = en;
        bool isCont(en=="9980"s);
        book(_h[en+"S"], 1, 1, 1+isCont);
        book(_h[en+"T"], 2, 1, 1+isCont);
      }
      raiseBeamErrorIf(_sqs.empty());
    }

    /// Recursively walk the decay tree to find the stable decay products of @a p
    void findDecayProducts(const Particle& mother, Particles& charged, Particles& neutral) const {
      for(const Particle& p : mother.children()) {
        if (!p.children().empty()) {
          findDecayProducts(p, charged, neutral);
        }
        else {
          if (isCharged(p))  charged.push_back(p);
          else               neutral.push_back(p);
        }
      }
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // Find the Upsilons among the unstables
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      Particles upsilons = ufs.particles(Cuts::pid==553 or Cuts::pid==100553);
      if (upsilons.empty() && _sqs == "9980"s) {
        Particles charged = apply<ChargedFinalState>(event, "CFS").particles();
        // at least 6 charged particles
        if (charged.size()<6) vetoEvent;
        // cut on high momentum particles
        size_t nHigh(0);
        for (const Particle & p : charged) {
          if (p.mom().p3().mod()>2.5) ++nHigh;
        }
        if (nHigh>1) vetoEvent;
        MSG_DEBUG("No Upsilons found => continuum event");
        Thrust thrust = apply<Thrust>(event, "Thrust");
        _h[_sqs+"T"]->fill(thrust.thrust());
        Sphericity sphericity = apply<Sphericity>(event, "Sphericity");
        _h[_sqs+"S"]->fill(sphericity.sphericity());
      }
      else {
        for (const Particle& ups : upsilons) {
          LorentzTransform boost;
          if (ups.p3().mod() > 1*MeV) {
            boost = LorentzTransform::mkFrameTransformFromBeta(ups.mom().betaVec());
          }
          // Find the decay products we want
          Particles charged,neutral;
          // 6 charged particles
          findDecayProducts(ups, charged, neutral);
          if (charged.size()<6) continue;
          // at most 1 |p|>2.5
          vector<FourMomentum> mom;
          mom.reserve(neutral.size()+charged.size());
          unsigned int nHigh(0);
          for (const Particle& p : charged) {
            mom.push_back(boost.transform(p.mom()));
            if(mom.back().p3().mod()>2.5) ++nHigh;
          }
          if (nHigh>1) continue;
          for (const Particle & p : neutral) {
            mom.push_back(boost.transform(p.mom()));
          }
          Thrust thrust;
          thrust.calc(mom);
          _h[_sqs+"T"]->fill(thrust.thrust());
          Sphericity sphericity;
          sphericity.calc(mom);
          _h[_sqs+"S"]->fill(sphericity.sphericity());
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      normalize(_h);
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(ARGUS_1986_I227324);

}
